/* Skript zur Ermittlung des kritischen Pfads */
-- Teil 1: Erstellung des Datenmodells und Einfuegen der Daten:
-- Skript zum konditionalen Loeschen eventuell bereits vorhandener Tabellen
declare
  cursor table_cur is
    select table_name
      from user_tables
     where table_name in ('PROCESS', 'PROCESS_HIERARCHY');
begin
  for tbl in table_cur loop
    execute immediate 'drop table ' || tbl.table_name || ' cascade constraints';
  end loop;
end;
/

-- Erzeugung der Tabellen
create table process (
  id char(1),
  run_duration number,
  min_start number,
  max_start number,
  min_end number,
  max_end number,
  critical_path_flag char(1),
  constraint pk_process primary key (id));
  
create table process_hierarchy(
  parent_id char(1),
  child_id char(1),
  constraint pk_process_hierarchy primary key (parent_id, child_id),
  constraint fk_parent foreign key (parent_id) references process(id),
  constraint fk_child foreign key (child_id) references process(id));
  
create index idx_process_hierarchy on process_hierarchy(child_id, parent_id);


-- Einfuegen der Daten
insert into process (id, run_duration)
select 'A', 3 from dual union all
select 'B', 4 from dual union all
select 'C', 2 from dual union all
select 'E', 5 from dual union all
select 'F', 5 from dual union all
select 'G', 3 from dual union all
select 'H', 7 from dual union all
select 'I', 6 from dual union all
select 'J', 8 from dual union all
select 'K', 2 from dual union all
select 'L', 1 from dual union all
select 'M', 4 from dual union all
select 'N', 3 from dual union all
select 'O', 2 from dual union all
select 'P', 11 from dual union all
select 'Q', 7 from dual union all
select 'R', 1 from dual union all
select 'S', 2 from dual union all
select 'T', 10 from dual union all
select 'U', 8 from dual union all
select 'V', 4 from dual union all
select 'X', 3 from dual union all
select 'Z', 4 from dual;

insert into process_hierarchy(parent_id, child_id)
select 'A', 'B' from dual union all
select 'A', 'C' from dual union all
select 'A', 'E' from dual union all
select 'A', 'G' from dual union all
select 'B', 'J' from dual union all
select 'H', 'J' from dual union all
select 'C', 'H' from dual union all
select 'E', 'H' from dual union all
select 'G', 'H' from dual union all
select 'G', 'I' from dual union all
select 'J', 'L' from dual union all
select 'H', 'F' from dual union all
select 'I', 'M' from dual union all
select 'M', 'N' from dual union all
select 'L', 'K' from dual union all
select 'F', 'K' from dual union all
select 'N', 'K' from dual union all
select 'L', 'Z' from dual union all
select 'K', 'O' from dual union all
select 'O', 'P' from dual union all
select 'O', 'T' from dual union all
select 'P', 'Q' from dual union all
select 'Q', 'R' from dual union all
select 'R', 'S' from dual union all
select 'T', 'U' from dual union all
select 'U', 'S' from dual union all
select 'Z', 'V' from dual union all
select 'S', 'V' from dual union all
select 'V', 'X' from dual;

commit;

-- Berechnung des kritischen Pfades
-- 1. Teilschritt: Darstellung der hierarchischen Beziehungen
 select parent_id, child_id, 
        sys_connect_by_path(parent_id, '|') || '|' 
           || child_id || '|' path
   from process_hierarchy ph
connect by prior ph.child_id = ph.parent_id
  start with ph.parent_id = 'A';

-- Zweiter Schritt: Aufsummieren der einzelnen Ausführungszeiten
select h.path, sum(p.run_duration) sum_duration
  from ( select parent_id, child_id, 
                sys_connect_by_path(parent_id, '|') || '|'
                   || child_id || '|' path
           from process_hierarchy ph
        connect by prior ph.child_id = ph.parent_id
          start with ph.parent_id = 'A') h
  join process p
    on instr(h.path, '|' || p.id || '|') > 0
 group by h.path
 order by sum_duration desc;
 

-- Dritter Schritt: Darstellung des kritischen Pfads
with critical_path as (
  select h.path, sum(p.run_duration) sum_duration,
         rank() over 
           (order by sum(p.run_duration) desc) rang
    from ( select parent_id, child_id, 
                  sys_connect_by_path(parent_id, '|') || '|' 
                     || child_id || '|' path
             from process_hierarchy ph
          connect by prior ph.child_id = ph.parent_id
            start with ph.parent_id = 'A') h
    join process p
      on instr(path, '|' || p.id || '|') > 0
   group by h.path)
select *
  from critical_path
 where rang = 1;
 

/* Loesung des Problems mit ISO-Standardmitteln */
with tasks (parent_id, child_id, run_duration, cum_end_time)
     as (
       select null as parent_id, o.id as child_id,
              o.run_duration, o.run_duration cum_end_time
         from process o
        where o.id = 'A'
       union all
       select ph.parent_id, ph.child_id, p.run_duration,
              t.cum_end_time + p.run_duration
         from process p
         join process_hierarchy ph on p.id = ph.child_id
         join tasks t on ph.parent_id = t.child_id)
select t.*, 
       max(cum_end_time - run_duration) over (partition by child_id) min_start_time,
       max(cum_end_time) over (partition by child_id) min_end_time,
       
  from tasks t
 order by parent_id nulls first, child_id;
 